<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ProductDetailsModel;
use App\Models\ProductPaymentModel;
use App\Models\PurchaseModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class MobPurchasePaymentController extends Controller
{
    public function index(Request $request)
    {
        try {
            $branchId = $request->branch_id;
            
            $purchases = PurchaseModel::with([
                'productDetails.product',
                'productDetails.suppliers',
                'payments'
            ])
            ->where('branch_id', $branchId)
            ->orderBy('id', 'desc')
            ->get();

            $processedData = $purchases->map(function($purchase) {
                // Calculate payment totals
                $totalPaidAmount = $purchase->payments()
                    ->whereNotIn('type', ['refund', 'loss'])
                    ->sum('paid_amount');
                
                $totalRefunded = $purchase->payments()
                    ->where('type', 'refund')
                    ->sum('paid_amount');
                
                $totalLoss = $purchase->payments()
                    ->where('type', 'loss')
                    ->sum('paid_amount');
                
                $netPaidAmount = $totalPaidAmount - $totalRefunded - $totalLoss;
                $dueAmount = $purchase->total_amount - $netPaidAmount;
                
                // Determine status
                $status = 'Unpaid';
                if ($netPaidAmount >= $purchase->total_amount) {
                    $status = 'Paid';
                } elseif ($netPaidAmount > 0) {
                    $status = 'Partially Paid';
                }

                // Get unique suppliers for this purchase
                $suppliers = $purchase->productDetails->pluck('suppliers.supplier_name')->filter()->unique();
                
                return [
                    'purchase_id' => $purchase->id,
                    'purchase_code' => $purchase->purchase_code,
                    'purchase_date' => $purchase->purchase_date->format('Y-m-d'),
                    'suppliers' => $suppliers->implode(', ') ?: 'N/A',
                    'total_amount' => round($purchase->total_amount, 2),
                    'paid_amount' => round($netPaidAmount, 2),
                    'due_amount' => round($dueAmount, 2),
                    'status' => $status,
                    'products_count' => $purchase->productDetails->count(),
                    'payments_count' => $purchase->payments->count(),
                    'latest_payment_date' => $purchase->payments->first()?->payment_date,
                    'notes' => $purchase->notes,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $processedData,
                'total_records' => $processedData->count()
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching purchase payments: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch purchase payments',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getPurchasesForPayment(Request $request)
    {
        try {
            $branchId = $request->branch_id;
            
            // Get purchases with pending payments (not fully paid)
            $purchases = PurchaseModel::with(['productDetails.product', 'productDetails.suppliers'])
                ->where('branch_id', $branchId)
                ->whereIn('status', ['pending', 'partial'])
                ->orderBy('purchase_date', 'desc')
                ->get();

            $processedData = $purchases->map(function($purchase) {
                // Calculate payment totals
                $totalPaidAmount = $purchase->payments()
                    ->whereNotIn('type', ['refund', 'loss'])
                    ->sum('paid_amount');
                
                $totalRefunded = $purchase->payments()
                    ->where('type', 'refund')
                    ->sum('paid_amount');
                
                $totalLoss = $purchase->payments()
                    ->where('type', 'loss')
                    ->sum('paid_amount');
                
                $netPaidAmount = $totalPaidAmount - $totalRefunded - $totalLoss;
                $dueAmount = $purchase->total_amount - $netPaidAmount;
                
                // Get unique suppliers for this purchase
                $suppliers = $purchase->productDetails->pluck('suppliers.supplier_name')->filter()->unique();

                return [
                    'purchase_id' => $purchase->id,
                    'purchase_code' => $purchase->purchase_code,
                    'purchase_date' => $purchase->purchase_date->format('Y-m-d'),
                    'suppliers' => $suppliers->implode(', ') ?: 'N/A',
                    'products_count' => $purchase->productDetails->count(),
                    'total_amount' => round($purchase->total_amount, 2),
                    'paid_amount' => round($netPaidAmount, 2),
                    'due_amount' => round(max(0, $dueAmount), 2),
                ];
            })->where('due_amount', '>', 0);

            return response()->json([
                'success' => true,
                'data' => $processedData->values()
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching purchases for payment: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch purchases for payment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'purchase_id' => 'required|integer|exists:purchases,id',
            'payment_type' => 'required|string|in:cash,upi,online,card',
            'payment_date' => 'required|date',
            'total_amount' => 'required|numeric|min:0',
            'paid_amount' => 'required|numeric|min:0.01',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            // Create payment record linked to purchase
            $payment = ProductPaymentModel::create([
                'product_dtl_id' => null, // Not linked to specific product detail
                'purchase_id' => $request->purchase_id, // 🔥 Link to purchase
                'payment_type' => $request->payment_type,
                'payment_date' => $request->payment_date,
                'total_amount' => $request->total_amount,
                'paid_amount' => $request->paid_amount,
                'type' => 'purchase_payment',
                'active_flag' => 'A'
            ]);

            // Update purchase status
            $this->updatePurchaseStatus($request->purchase_id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Payment created successfully',
                'payment_id' => $payment->id
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating purchase payment: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to create payment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show(Request $request, $id)
    {
        try {
            $purchase = PurchaseModel::with([
                'productDetails.product',
                'productDetails.suppliers',
                'payments' => function($query) {
                    $query->orderBy('payment_date', 'desc');
                }
            ])->where('id', $id)->first();

            if (!$purchase) {
                return response()->json([
                    'success' => false,
                    'message' => 'Purchase not found'
                ], 404);
            }

            // Calculate payment totals
            $totalPaidAmount = $purchase->payments()
                ->whereNotIn('type', ['refund', 'loss'])
                ->sum('paid_amount');
            
            $totalRefunded = $purchase->payments()
                ->where('type', 'refund')
                ->sum('paid_amount');
            
            $totalLoss = $purchase->payments()
                ->where('type', 'loss')
                ->sum('paid_amount');
            
            $netPaidAmount = $totalPaidAmount - $totalRefunded - $totalLoss;
            $dueAmount = $purchase->total_amount - $netPaidAmount;

            $data = [
                'purchase_details' => [
                    'id' => $purchase->id,
                    'purchase_code' => $purchase->purchase_code,
                    'purchase_date' => $purchase->purchase_date->format('Y-m-d'),
                    'total_amount' => round($purchase->total_amount, 2),
                    'status' => $purchase->status,
                    'notes' => $purchase->notes,
                ],
                'payment_summary' => [
                    'total_amount' => round($purchase->total_amount, 2),
                    'paid_amount' => round($netPaidAmount, 2),
                    'due_amount' => round($dueAmount, 2),
                    'refunded_amount' => round($totalRefunded, 2),
                    'loss_amount' => round($totalLoss, 2),
                ],
                'products' => $purchase->productDetails->map(function($detail) {
                    return [
                        'product_detail_id' => $detail->product_details_sid,
                        'product_name' => $detail->name,
                        'product_code' => $detail->code,
                        'supplier_name' => $detail->suppliers->supplier_name ?? 'N/A',
                        'qty' => $detail->qty,
                        'unit_price' => $detail->unit_price,
                        'margin_price' => $detail->margin_price,
                        'total_amount' => round($detail->total_amount, 2),
                    ];
                }),
                'payments' => $purchase->payments->map(function($payment) {
                    return [
                        'id' => $payment->id,
                        'payment_type' => $payment->payment_type,
                        'payment_date' => $payment->payment_date,
                        'paid_amount' => round($payment->paid_amount, 2),
                        'type' => $payment->type,
                        'created_at' => $payment->created_at->format('Y-m-d H:i:s'),
                    ];
                })
            ];

            return response()->json([
                'success' => true,
                'data' => $data
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching purchase payment details: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch payment details',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function addPayment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'purchase_id' => 'required|integer|exists:purchases,id',
            'payment_type' => 'required|string|in:cash,upi,online,card',
            'payment_date' => 'required|date',
            'paid_amount' => 'required|numeric|min:0.01',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $purchase = PurchaseModel::findOrFail($request->purchase_id);
            
            // Calculate current due amount
            $totalPaidAmount = $purchase->payments()
                ->whereNotIn('type', ['refund', 'loss'])
                ->sum('paid_amount');
            
            $totalRefunded = $purchase->payments()
                ->where('type', 'refund')
                ->sum('paid_amount');
            
            $totalLoss = $purchase->payments()
                ->where('type', 'loss')
                ->sum('paid_amount');
            
            $netPaidAmount = $totalPaidAmount - $totalRefunded - $totalLoss;
            $dueAmount = $purchase->total_amount - $netPaidAmount;

            if ($request->paid_amount > $dueAmount) {
                return response()->json([
                    'success' => false,
                    'message' => "Payment amount cannot exceed due amount of ₹{$dueAmount}"
                ], 400);
            }

            // Create new payment linked to purchase
            ProductPaymentModel::create([
                'product_dtl_id' => null, // Not linked to specific product detail
                'purchase_id' => $request->purchase_id, // 🔥 Link to purchase
                'payment_type' => $request->payment_type,
                'payment_date' => $request->payment_date,
                'total_amount' => $purchase->total_amount,
                'paid_amount' => $request->paid_amount,
                'type' => 'purchase_payment',
                'active_flag' => 'A'
            ]);

            // Update purchase status
            $this->updatePurchaseStatus($request->purchase_id);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Payment added successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error adding payment: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to add payment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    private function updatePurchaseStatus($purchaseId)
    {
        $purchase = PurchaseModel::findOrFail($purchaseId);
        
        // Calculate payment totals
        $totalPaidAmount = $purchase->payments()
            ->whereNotIn('type', ['refund', 'loss'])
            ->sum('paid_amount');
        
        $totalRefunded = $purchase->payments()
            ->where('type', 'refund')
            ->sum('paid_amount');
        
        $totalLoss = $purchase->payments()
            ->where('type', 'loss')
            ->sum('paid_amount');
        
        $netPaidAmount = $totalPaidAmount - $totalRefunded - $totalLoss;
        $dueAmount = $purchase->total_amount - $netPaidAmount;

        // Determine purchase status
        $status = 'pending';
        if ($netPaidAmount >= $purchase->total_amount) {
            $status = 'paid';
        } elseif ($netPaidAmount > 0) {
            $status = 'partial';
        }

        // Update purchase
        $purchase->update([
            'paid_amount' => $netPaidAmount,
            'due_amount' => $dueAmount,
            'status' => $status
        ]);

        // Update product details status (linked to this purchase)
        $productStatus = 1; // Unpaid
        if ($status === 'paid') {
            $productStatus = 2; // Fully Paid
        } elseif ($status === 'partial') {
            $productStatus = 3; // Partially Paid
        }

        $purchase->productDetails()->update(['status' => $productStatus]);
    }

    public function delete($id)
    {
        try {
            DB::beginTransaction();

            $purchase = PurchaseModel::findOrFail($id);
            
            // Delete all payments for this purchase
            ProductPaymentModel::where('purchase_id', $id)->delete();

            // Reset purchase status to pending
            $purchase->update([
                'paid_amount' => 0,
                'due_amount' => $purchase->total_amount,
                'status' => 'pending'
            ]);

            // Reset product details status to unpaid
            $purchase->productDetails()->update(['status' => 1]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'All payments deleted successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting payments: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete payments',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
