<?php

namespace App\Http\Controllers;

use App\Models\InvoiceItemModel;
use App\Models\OrdersItemsModel;
use App\Models\PaymentModeModel;
use App\Models\PaymentStatusModel;
use App\Models\ProductDetailsModel;
use App\Models\ProductHistoryModel;
use App\Models\ProductPaymentModel;
use App\Models\ProductModel;
use App\Models\CategoryModel;
use App\Models\ProductTypeModel;
use App\Models\QuotesItemsModel;
use App\Models\SupplierModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;


class ProductController extends Controller
{
    protected $branch_id;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->branch_id = session('branch_id');
            return $next($request);
        });
    }

    public function index(Request $request)
    {
        $products = ProductModel::with('user','branch')->where('branch_id', $this->branch_id)->orderBy('id', 'desc')
            ->get();

        return view('products.index', compact('products'));
    }
    public function add(Request $request)
    {
        // $categories = CategoryModel::where('branch_id', $this->branch_id)->get();
        $categories = CategoryModel::get();
        $suppliers = SupplierModel::where('branch_id', $this->branch_id)->where('active_status', 1)->get();
        $payment_type = PaymentModeModel::get();
        $product_type = ProductTypeModel::where('oprntl_flag', 'A')->get();

        return view('products.add', compact('categories', 'suppliers', 'payment_type', 'product_type'));
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'supplier_id' => 'required',
            'payment_date' => 'nullable|date',
            'product.*.name' => 'required',
            // 'product.*.productCode' => 'required|unique:products,code,NULL,id,branch_id,' . session('branch_id'),
            'product.*.productCode' => 'required',
            'product.*.unit_price' => 'required|numeric',
            'product.*.category_id' => 'required',
            'product.*.qty' => 'required|integer',
            'product.*.margin_price' => 'required|numeric',
            'product.*.product_type' => 'required|exists:product_type,id',
            'product.*.payment_type' => 'required_with:product.*.paid_amount',
            'product.*.paid_amount' => 'nullable|numeric|lte:product.*.total_amount',
        ]);

        if ($validator->fails()) {

            $errors = [];
            foreach ($validator->errors()->messages() as $field => $messages) {
                if (str_contains($field, 'product.')) {
                    preg_match('/product\.(\d+)\.(.+)/', $field, $matches);
                    $rowIndex = $matches[1];
                    $fieldName = $matches[2];
                    $errors['rows'][$rowIndex][$fieldName] = $messages[0];
                } else {
                    $errors['form'][$field] = $messages[0];
                }
            }

            return response()->json([
                'status' => 'error',
                'errors' => $errors
            ], 422);
        }

        DB::beginTransaction();
        try {
            foreach ($request->product as $data) {

                $details_code = $this->generateProductDetailCode();

                $product_path = null;
                if (isset($data['product_image']) && $data['product_image']) {
                    $product_path = storeFile($data['product_image'], 'products');
                }

                $status = 1;
                if ($data['paid_amount']) {
                    $paid = $data['paid_amount'];
                    $status = ($data['total_amount'] == $paid) ? 2 : 3;
                }

                $product = ProductModel::create([
                    'name' => $data['name'],
                    'native_name' => $data['native_name'],
                    'code' => $data['productCode'],
                    'hsn_code' => $data['hsnCode'],
                    'category_id' => $data['category_id'],
                    'supplier_id' => $request->supplier_id,
                    'unit_price' => $data['unit_price'],
                    'description' => $data['description'] ?? null,
                    'products_image' => $product_path,
                    'branch_id' => $this->branch_id,
                    'margin_price' => $data['margin_price'],
                    'qty' => $data['qty'],
                    'total_qty' => $data['qty'],
                    'product_type' => $data['product_type'],
                ]);

                $product_id = $product->id;

                $productDetail = ProductDetailsModel::create([
                    'product_id' => $product_id,
                    'product_details_code' => $details_code,
                    'name' => $data['name'],
                    'native_name' => $data['native_name'],
                    'code' => $data['productCode'],
                    'category_id' => $data['category_id'],
                    'supplier_id' => $request->supplier_id,
                    'unit_price' => $data['unit_price'],
                    'margin_price' => $data['margin_price'],
                    'description' => $data['description'] ?? null,
                    'products_image' => $product_path,
                    'branch_id' => $this->branch_id,
                    'qty' => $data['qty'],
                    'total_amount' => $data['total_amount'],
                    'status' => $status
                ]);

                // Save Payments
                if (!empty($productDetail)) {
                    if (!empty($data['paid_amount'])) {
                        ProductPaymentModel::create([
                            'product_dtl_id' => $productDetail->product_details_sid,
                            'payment_type' => $data['payment_type'],
                            'payment_date' => $request->payment_date ?? now(),
                            'total_amount' => $data['total_amount'],
                            'paid_amount' => $data['paid_amount'],
                            'type' => 'paid',
                        ]);
                    }
                }

                // History Record
                ProductHistoryModel::create([
                    'product_id' => $product_id,
                    'branch_id' => $this->branch_id,
                    'name' => $data['name'],
                    'native_name' => $data['native_name'],
                    'code' => $data['productCode'],
                    'category_id' => $data['category_id'],
                    'supplier_id' => $request->supplier_id,
                    'unit_price' => $data['unit_price'],
                    'qty' => $data['qty'],
                    'description' => $data['description'] ?? null,
                    'product_image' => $product_path,
                    'action' => 'A',
                    'created_by' => auth()->user()->id,
                ]);
            }

            DB::commit();
            return response()->json([
                'status' => 'success',
                'message' => 'Product added successfully!'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Product Store Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong!',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    public function edit($id)
    {
        $product = ProductModel::with('details.suppliers', 'details.paymentStatus', 'details.payments')->findOrFail($id);

        $product->details = $product->details->sortByDesc('product_details_code')->values();

        // $categories = CategoryModel::where('branch_id', $this->branch_id)->get();
        $categories = CategoryModel::get();

        $suppliers = SupplierModel::where('branch_id', $this->branch_id)->where('active_status', 1)->get();

        return view('products.edit', compact('product', 'categories', 'suppliers'));
    }

    public function purchase()
    {
        $product = ProductModel::with('category')->where('branch_id', session('branch_id'))->whereIn('product_type', [2, 3])->doesntHave('assembledItem')->get();

        $suppliers = SupplierModel::where('branch_id', $this->branch_id)->get();

        // $category = CategoryModel::where('branch_id', session('branch_id'))->get();
        $category = CategoryModel::get();

        $payment_type = PaymentModeModel::get();

        return view('products.purchase', compact('product', 'suppliers', 'payment_type', 'category'));
    }

    public function savePurchase(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product.*.product_id' => 'required',
            'payment_date' => 'nullable|date',
            'product.*.unit_price' => 'required|numeric',
            'product.*.margin_price' => 'required|numeric',
            'product.*.qty' => 'required|integer',
            'product.*.payment_type' => 'required_with:product.*.paid_amount',
            'product.*.paid_amount' => 'nullable|numeric|lte:product.*.total_amount',
            'supplier_id' => 'required',
        ]);

        if ($validator->fails()) {
            $errors = [];
            foreach ($validator->errors()->messages() as $field => $messages) {
                if (str_contains($field, 'product.')) {
                    preg_match('/product\.(\d+)\.(.+)/', $field, $matches);
                    $rowIndex = $matches[1];
                    $fieldName = $matches[2];
                    $errors['rows'][$rowIndex][$fieldName] = $messages[0];
                } else {
                    $errors['form'][$field] = $messages[0];
                }
            }

            return response()->json([
                'status' => 'error',
                'errors' => $errors
            ], 422);
        }

        DB::beginTransaction();
        try {
            foreach ($request->product as $data) {
                if (is_numeric($data['product_id'])) {
                    $product = ProductModel::findOrFail($data['product_id']);

                    $product->update([
                        'qty' => $product->qty + $data['qty'],
                        'total_qty' => $product->total_qty + $data['qty'],
                        'unit_price' => $data['unit_price'],
                        'margin_price' => $data['margin_price'],
                    ]);
                } else {
                    $product = ProductModel::create([
                        'name' => $data['product_id'],
                        'code' => $data['productCode'],
                        'hsn_code' => $data['hsnCode'],
                        'category_id' => $data['category_id'],
                        'supplier_id' => $request->supplier_id,
                        'unit_price' => $data['unit_price'],
                        'description' => $data['description'] ?? null,
                        'branch_id' => $this->branch_id,
                        'margin_price' => $data['margin_price'],
                        'qty' => $data['qty'],
                        'total_qty' => $data['qty'],
                        'product_type' => 3,
                    ]);
                }

                $details_code = $this->generateProductDetailCode();

                $status = 1;
                if ($data['paid_amount']) {
                    $status = ($data['total_amount'] == $data['paid_amount']) ? 2 : 3;
                }

                $purchase = ProductDetailsModel::create([
                    'product_id' => $product->id,
                    'product_details_code' => $details_code,
                    'branch_id' => $this->branch_id,
                    'name' => $product->name,
                    'code' => $product->code,
                    'category_id' => $product->category_id,
                    'supplier_id' => $request->supplier_id,
                    'unit_price' => $data['unit_price'],
                    'margin_price' => $data['margin_price'],
                    'qty' => $data['qty'],
                    'status' => $status,
                    'total_amount' => $data['total_amount'],
                    'description' => $data['description'] ?? null,
                ]);

                if (!empty($data['paid_amount'])) {
                    ProductPaymentModel::create([
                        'product_dtl_id' => $purchase->product_details_sid,
                        'payment_type' => $data['payment_type'],
                        'payment_date' => $request->payment_date ?? now(),
                        'total_amount' => $data['total_amount'],
                        'paid_amount' => $data['paid_amount'],
                        'type' => 'paid',
                    ]);
                }
            }

            DB::commit();
            return response()->json([
                'status' => 'success',
                'message' => 'Purchased products added successfully!'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('While adding Purchase for the product: ' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong!',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    public function update(Request $request)
    {
        $request->validate([
            'name' => 'required',
            // 'productCode' => 'required|unique:products,code,' . $request->product_id . ',id,branch_id,' . session('branch_id'),
            'productCode' => 'required',
            'unit_price' => 'required|numeric',
            'category_id' => 'required',
            'supplier_id' => 'required',
            'qty' => 'required|numeric',
            'margin_price' => 'required|numeric',
        ]);
       
        DB::beginTransaction();
        try {
            $product = ProductModel::findOrFail($request->product_id);

            // Handle product image update
            $product_path = $product->products_image;
            if ($request->hasFile('product_image')) {
                $product_path = storeFile($request->file('product_image'), 'products');

                if ($product->products_image && Storage::disk('s3')->exists($product->products_image)) {
                    Storage::disk('s3')->delete($product->products_image);
                }
            }

            $quantityDifference = $request->qty - $product->qty;
            $newTotalQuantity = $product->total_qty + $quantityDifference;

            if ($newTotalQuantity < 0) {
                throw new \Exception("Total quantity cannot be negative");
            }

            $product->update([
                'name' => $request->name,
                'native_name' => $request->native_name,
                'code' => $request->productCode,
                'hsn_code' => $request->hsnCode,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'margin_price' => $request->margin_price,
                'description' => $request->description,
                'products_image' => $product_path,
                'qty' => $request->qty,
                'total_qty' => $newTotalQuantity,
            ]);

            ProductHistoryModel::create([
                'product_id' => $request->product_id,
                'branch_id' => $this->branch_id,
                'name' => $request->name,
                'native_name' => $request->native_name,
                'code' => $request->productCode,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'qty' => $request->qty,
                'description' => $request->description,
                'product_image' => $product_path,
                'action' => 'E',
                'created_by' => auth()->user()->id,
            ]);

            DB::commit();
            return redirect()->route('product-index')->with('success', 'Product updated successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('While Updating the product: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage())->withInput();
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();

        try {
            $product = ProductModel::with(['details.payments'])
                ->where('active_flag', 'A')
                ->find($id);

            if (!$product) {
                return redirect()->back()->with('error', 'Product not found or already inactive!');
            }

            $usedIn = [];

            if (InvoiceItemModel::where('product_id', $product->id)->where('active_flag', 'A')->exists()) {
                $usedIn[] = 'invoices';
            }

            if (OrdersItemsModel::where('product_id', $product->id)->where('active_flag', 'A')->exists()) {
                $usedIn[] = 'orders';
            }

            if (QuotesItemsModel::where('product_id', $product->id)->where('active_flag', 'A')->exists()) {
                $usedIn[] = 'quotes';
            }

            $productDetailIds = $product->details->pluck('product_details_sid');

            if (
                $productDetailIds->isNotEmpty() &&
                ProductPaymentModel::whereIn('product_dtl_id', $productDetailIds)
                    ->where('active_flag', 'A')->exists()
            ) {
                $usedIn[] = 'payments';
            }

            if (!empty($usedIn)) {
                DB::rollBack();
                return redirect()->back()->with('error', 'Cannot delete product: It exists in active ' . implode(', ', $usedIn) . '.');
            }

            // Create history record
            ProductHistoryModel::create([
                'product_id' => $product->id,
                'branch_id' => $this->branch_id,
                'name' => $product->name,
                'code' => $product->code,
                'category_id' => $product->category_id,
                'supplier_id' => $product->supplier_id,
                'unit_price' => $product->unit_price,
                'qty' => $product->qty,
                'description' => $product->description,
                'product_image' => $product->products_image,
                'action' => 'D',
                'created_by' => auth()->id(),
            ]);

            foreach ($product->details as $detail) {
                if ($detail->payments->isNotEmpty()) {
                    $detail->payments->each->delete();
                }
                $detail->delete();
            }

            // Delete main product
            $product->delete();

            DB::commit();

            return redirect()->route('product-index')->with('success', 'Product deleted successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error("Product deletion failed for ID {$id}: " . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete product');
        }
    }

    private function generateProductDetailCode(): string
    {
        $lastDetail = ProductDetailsModel::orderBy('product_details_sid', 'desc')->first();

        if ($lastDetail && $lastDetail->product_details_code) {
            $lastCode = $lastDetail->product_details_code;
            $parts = explode('-', $lastCode);
            $prefix = $parts[0];
            $number = isset($parts[1]) ? (int) $parts[1] : 0;
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }

        return 'PROD-001';
    }

    public function detailsUpdate(Request $request)
    {
        $request->validate([
            'detail_id' => 'required|exists:product_details,product_details_sid',
            'purchased_qty' => 'required|numeric|min:0',
            'status' => 'required|in:refund,update,non_returnable'
        ]);

        DB::beginTransaction();
        try {
            $details = ProductDetailsModel::with('payments', 'paymentStatus')
                ->where('product_details_sid', $request->detail_id)
                ->firstOrFail();
            $product = ProductModel::where('id', $details->product_id)->first();

            $currentQty = $details->qty;
            $newQty = (float) $request->purchased_qty;
            $marginPrice = $details->margin_price;
            $newTotalAmount = $marginPrice * $newQty;
            $totalPaid = $details->payments->whereNotIn('type', ['refund', 'loss'])->sum('paid_amount');
            $currentTotal = $details->total_amount;

            switch ($request->status) {
                case 'refund':
                    $this->processRefund($details, $currentQty, $newQty, $marginPrice, $newTotalAmount, $totalPaid, $request, $product, $currentTotal);
                    break;

                case 'update':
                    $this->processPaid($details, $currentQty, $newQty, $marginPrice, $newTotalAmount, $totalPaid, $request, $product);
                    break;

                case 'non_returnable':
                    $this->processLoss($details, $currentQty, $newQty, $marginPrice, $newTotalAmount, $totalPaid, $request, $product, $currentTotal);
                    break;
            }
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Product detail updated successfully.'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to update product details: ' . $e->getMessage()
            ], 500);
        }
    }

    protected function processRefund($details, $currentQty, $newQty, $marginPrice, $newTotalAmount, $totalPaid, $request, $product, $currentTotal)
    {
        if ($newQty >= $currentQty) {
            throw new \Exception('For refund, entered quantity must be less than current quantity.');
        }

        $refundQty = $currentQty - $newQty;

        if ($product->qty < $refundQty) {
            throw new \Exception('Refund quantity exceeds available stock.');
        }

        $refundAmount = $request->amount;

        $totalPaidAmount = ProductPaymentModel::where('product_dtl_id', $request->detail_id)
            ->whereNotIn('type', ['refund', 'loss'])
            ->sum('paid_amount');

        $totalRefunded = ProductPaymentModel::where('product_dtl_id', $request->detail_id)
            ->where('type', 'refund')
            ->sum('paid_amount');

        $availableRefundAmount = $totalPaidAmount - $totalRefunded;

        $originalUnitPrice = $currentTotal / $currentQty;
        $maxRefundByQty = $refundQty * $originalUnitPrice;

        if ($refundAmount > min($availableRefundAmount, $maxRefundByQty)) {
            $maxAllowed = min($availableRefundAmount, $maxRefundByQty);
            throw new \Exception("Refund amount ({$refundAmount}) exceeds maximum allowed ({$maxAllowed})");
        }

        $paymentStatusName = $details->paymentStatus->name ?? 'Unpaid';

        if ($paymentStatusName !== 'Unpaid' && $refundAmount > 0) {
            ProductPaymentModel::create([
                'product_dtl_id' => $request->detail_id,
                'payment_type' => $request->payment_type ?? 'cash',
                'payment_date' => now(),
                'total_amount' => $currentTotal,
                'paid_amount' => $refundAmount,
                'type' => 'refund',
            ]);

            $netPaidAmount = $totalPaidAmount - ($totalRefunded + $refundAmount);
            $remainingTotal = $newQty * $originalUnitPrice;

            if ($netPaidAmount == $remainingTotal) {
                $updatedStatus = 'Paid';
            } elseif ($netPaidAmount > 0) {
                $updatedStatus = 'Partially Paid';
            } else {
                $updatedStatus = 'Unpaid';
            }

            $status_id = PaymentStatusModel::where('name', $updatedStatus)->value('id');
        } else {
            $status_id = $details->status;
        }

        $details->update([
            'qty' => $newQty,
            'total_amount' => $newTotalAmount,
            'status' => $status_id,
        ]);

        $product->update([
            'qty' => $product->qty - $refundQty,
        ]);
    }

    protected function processPaid($details, $currentQty, $newQty, $marginPrice, $newTotalAmount, $totalPaid, $request, $product)
    {
        if ($currentQty >= $newQty) {
            throw new \Exception(message: 'For payment, new quantity must be greater than current quantity');
        }
        $diffQty = $newQty - $currentQty;

        $totalRefunded = ProductPaymentModel::where('product_dtl_id', $request->detail_id)
            ->where('type', 'refund')
            ->sum('paid_amount');

        $newPaidAmount = ($totalPaid - $totalRefunded) + $request->amount;

        if ($request->amount > 0) {
            ProductPaymentModel::create([
                'product_dtl_id' => $request->detail_id,
                'payment_type' => 'cash',
                'payment_date' => now(),
                'total_amount' => $newTotalAmount,
                'paid_amount' => $request->amount,
                'type' => 'paid',
            ]);
        }

        if ($newPaidAmount == $newTotalAmount) {
            $paymentStatus = 'Paid';
        } elseif ($newPaidAmount > 0) {
            $paymentStatus = 'Partially Paid';
        } else {
            $paymentStatus = 'Unpaid';
        }

        $status_id = PaymentStatusModel::where('name', $paymentStatus)->value('id');

        $details->update([
            'qty' => $newQty,
            'total_amount' => $newTotalAmount,
            'status' => $status_id,
        ]);

        $product->update([
            'qty' => $product->qty + $diffQty
        ]);
    }

    protected function processLoss($details, $currentQty, $newQty, $marginPrice, $newTotalAmount, $totalPaid, $request, $product, $currentTotal)
    {
        if ($newQty >= $currentQty) {
            throw new \Exception('For Non Returnable, new quantity must be less than current quantity.');
        }

        $lostQty = $currentQty - $newQty;

        if ($lostQty <= 0) {
            throw new \Exception('Non Returnable quantity must be greater than zero.');
        }

        if ($product->qty < $lostQty) {
            throw new \Exception('Non Returnable quantity exceeds available stock in main product.');
        }

        $originalUnitPrice = $currentTotal / $currentQty;
        $lossAmount = $marginPrice * $lostQty;

        $totalPaidAmount = ProductPaymentModel::where('product_dtl_id', $request->detail_id)
            ->whereNotIn('type', ['refund', 'loss'])
            ->sum('paid_amount');

        $totalRefunded = ProductPaymentModel::where('product_dtl_id', $request->detail_id)
            ->where('type', 'refund')
            ->sum('paid_amount');

        $totalLoss = ProductPaymentModel::where('product_dtl_id', $request->detail_id)
            ->where('type', 'loss')
            ->sum('paid_amount');

        $netPaidAmount = $totalPaidAmount - $totalRefunded - $totalLoss;
        $newTotal = $newQty * $originalUnitPrice;

        $paymentStatusName = $details->paymentStatus->name ?? 'Unpaid';
        if ($paymentStatusName !== 'Unpaid' && $lossAmount > 0) {
            ProductPaymentModel::create([
                'product_dtl_id' => $request->detail_id,
                'payment_type' => $request->payment_type ?? 'cash',
                'payment_date' => now(),
                'total_amount' => $currentTotal,
                'paid_amount' => $lossAmount,
                'type' => 'loss',
            ]);

            $netPaidAmount -= $lossAmount;
        }

        if ($netPaidAmount == $newTotal) {
            $updatedStatus = 'Paid';
        } elseif ($netPaidAmount > 0) {
            $updatedStatus = 'Partially Paid';
        } else {
            $updatedStatus = 'Unpaid';
        }

        $status_id = PaymentStatusModel::where('name', $updatedStatus)->value('id');

        $details->update([
            'qty' => $newQty,
            'total_amount' => $newTotal,
            'status' => $status_id,
        ]);

        $product->update([
            'qty' => $product->qty - $lostQty,
        ]);
    }



}